unit uTreeNode;

interface

uses Rubies, QComCtrls;

var
  cTreeNode, cTreeNodes: Tvalue;

function GetNode(obj: Tvalue): TTreeNode;
function ap_cTreeNode: Tvalue; stdcall;
procedure TreeNode_setup(obj: Tvalue; real: TTreeNode);
function TreeNode_alloc(This: Tvalue; real: TTreeNode): Tvalue; stdcall;
function ap_cTreeNodes: Tvalue; stdcall;
function TreeNodes_alloc(This: Tvalue; real: TTreeNodes): Tvalue; stdcall;
procedure Init_TreeNode;

implementation

uses
  SysUtils, Classes, uStrUtils,
  uHandle, uIntern, uAlloc, uProp, uPhi,
  uSizeConstraints,
  uComponent, uControl;

type
  PNode = ^TNode;
  TNode = record
    real: TTreeNode;
    dead: Boolean;
    data: Tvalue;
  end;

procedure TreeNode_free(p: PNode); cdecl;
begin
  dispose(p);
end;

procedure TreeNode_mark(p: PNode); cdecl;
begin
  rb_gc_mark(Pointer(p^.data));
end;

function TreeNode_alloc1(klass: Tvalue; real: TTreeNode): Tvalue; stdcall;
var
  p: PNode;
  parent: Tvalue;
begin
  if real = nil then begin result := Qnil; exit; end;
  new(p);
  p^.real := real;
  p^.dead := False;
  p^.data := Qnil;
  result := rb_data_object_alloc(klass, p, @TreeNode_mark, @TreeNode_free);
  rb_iv_set(result, '@events', rb_hash_new);
  real.Data := Pointer(result);
  if real.Parent = nil then
    parent := Qnil
  else
    parent := Tvalue(real.Parent.Data);
  rb_iv_set(result, '@parent', parent);
end;

function GetP(obj: Tvalue): PNode;
begin
  if rb_obj_is_instance_of(obj, cTreeNode) = 0 then
    ap_raise(ap_eArgError, sWrong_arg_type);
  result := ap_data_get_struct(obj);
end;

function GetNode(obj: Tvalue): TTreeNode;
var
  p: PNode;
begin
  p := GetP(obj);
  if p^.dead then
  begin
    ap_raise(ap_eArgError, 'dead node');
    result := nil;
  end
  else
    result := p^.real;
end;

function ap_cTreeNode: Tvalue; stdcall;
begin
  result := cTreeNode;
end;

procedure TreeNode_setup(obj: Tvalue; real: TTreeNode);
var
  i: Integer;
begin
  rb_iv_set(obj, '@text', rb_str_new2(PChar(WideCharToString(PWideChar(real.Text)))));
  rb_iv_set(obj, '@index', INT2FIX(real.Index));
  rb_iv_set(obj, '@level', INT2FIX(real.Level));
  for i := 0 to real.Count-1 do
    TreeNode_alloc(cTreeNode, real[i]);
end;

function TreeNode_alloc(This: Tvalue; real: TTreeNode): Tvalue; stdcall;
begin
  result := TreeNode_alloc1(This, real);
  TreeNode_setup(result, real);
end;

function TreeNode_aref(This, v: Tvalue): Tvalue; cdecl;
var
  real: TTreeNode;
  n: Integer;
begin
  real := GetNode(This);
  n := NUM2INT(v);
  if (n < 0) or (real.Count <= n) then
    ap_raise(ap_eIndexError, sOut_of_range);
  if real[n].Data = nil then
    result := Qnil
  else
    result := Tvalue(real[n].Data);
end;

function TreeNode_get_data(This: Tvalue): Tvalue; cdecl;
var
  p: PNode;
begin
  p := GetP(This);
  result := p^.data;
end;

function TreeNode_set_data(This, v: Tvalue): Tvalue; cdecl;
var
  p: PNode;
begin
  p := GetP(This);
  p^.data := v;
  result := v;
end;

function TreeNode_get_count(This: Tvalue): Tvalue; cdecl;
var
  real: TTreeNode;
begin
  real := GetNode(This);
  result := INT2FIX(real.Count);
end;

function TreeNode_set_text(This, v: Tvalue): Tvalue; cdecl;
var
  real: TTreeNode;
  str: Tvalue;
begin
  real := GetNode(This);
  real.Text := STR2CSTR(v);
  str := rb_iv_get(This, '@text');
  rb_str_resize(str, 0);
  rb_str_concat(str, v);
  result := v;
end;

function SortProc(a, b, This: Longint): Integer; stdcall;
var
  ret, anode, bnode: Tvalue;
begin
  anode := Tvalue(TTreeNode(a).Data);
  bnode := Tvalue(TTreeNode(b).Data);
  ret := rb_yield(rb_assoc_new(anode, bnode));
  if RTYPE(ret) = T_FIXNUM then
    result := FIX2INT(ret)
  else
    result := 0;
end;

function TreeNode_sort(This: Tvalue): Tvalue; cdecl;
var
  real: TTreeNode;
begin
  real := GetNode(This);
  result := ap_bool(real.AlphaSort(True));
end;

function TreeNode_assign(This, v: Tvalue): Tvalue; cdecl;
var
  real: TTreeNode;
  source: TPersistent;
begin
  real := GetNode(This);
  ap_data_get_object(v, TPersistent, source);
  real.Assign(source);
  result := v;
end;

function TreeNode_expand(This, recurse: Tvalue): Tvalue; cdecl;
var
  real: TTreeNode;
begin
  real := GetNode(This);
  real.Expand(RTEST(recurse));
  result := Qnil;
end;

function TreeNode_collapse(This, recurse: Tvalue): Tvalue; cdecl;
var
  real: TTreeNode;
begin
  real := GetNode(This);
  real.Collapse(RTEST(recurse));
  result := Qnil;
end;

function ap_cTreeNodes: Tvalue; stdcall;
begin
  result := cTreeNodes;
end;

procedure TreeNodes_setup(obj: Tvalue; real: TTreeNodes); stdcall;
var
  i: Integer;
begin
  rb_iv_set(obj, '@items', rb_ary_new);
  for i := 0 to real.Count-1 do
    TreeNode_alloc(cTreeNode, real[i]);
end;

function TreeNodes_alloc(This: Tvalue; real: TTreeNodes): Tvalue; stdcall;
begin
  result := TmpAlloc(This, real);
  TreeNodes_setup(result, real);
end;

function TreeNodes_aref(This, v: Tvalue): Tvalue; cdecl;
var
  n: Integer;
  real: TTreeNodes;
begin
  real := ap_data_get_struct(This);
  n := NUM2INT(v);
  if (n < 0) or (real.Count <= n) then
    ap_raise(ap_eIndexError, sOut_of_range);
  if real[n].Data = nil then
    result := Qnil
  else
  begin
    result := Tvalue(real[n].Data);
    GetNode(result);
  end;
end;

function TreeNodes_get_count(This: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.Count);
end;

function TreeNodes_get_first_node(This: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
begin
  real := ap_data_get_struct(This);
  if real.GetFirstNode.Data = nil then
    result := Qnil
  else
    result := Tvalue(real.GetFirstNode.Data);
end;

function TreeNodes_get_owner(This: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
begin
  real := ap_data_get_struct(This);
  if real.Owner = nil then
    result := Qnil
  else
    result := real.Owner.tag
  ;
end;

function TreeNodes_add(This, vnode, str: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
  node, ret: TTreeNode;
begin
  real := ap_data_get_struct(This);
  if vnode = Qnil then node := nil else node := GetNode(vnode);
  ret := real.Add(node, STR2CSTR(str));
  result := TreeNode_alloc(cTreeNode, ret);
  rb_iv_set(result, '@owner', This);
  rb_ary_push(rb_iv_get(This, '@items'), result);
end;

function TreeNodes_add_first(This, vnode, str: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
  node, ret: TTreeNode;
begin
  real := ap_data_get_struct(This);
  if vnode = Qnil then node := nil else node := GetNode(vnode);
  ret := real.Add(node, STR2CSTR(str));
  result := TreeNode_alloc(cTreeNode, ret);
  rb_iv_set(result, '@owner', This);
  rb_ary_push(rb_iv_get(This, '@items'), result);
end;

function TreeNodes_add_child(This, vnode, str: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
  node, ret: TTreeNode;
begin
  real := ap_data_get_struct(This);
  node := GetNode(vnode);
  ret := real.AddChild(node, STR2CSTR(str));
  result := TreeNode_alloc(cTreeNode, ret);
  rb_iv_set(result, '@owner', This);
  rb_ary_push(rb_iv_get(This, '@items'), result);
end;

function TreeNodes_add_child_first(This, vnode, str: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
  node, ret: TTreeNode;
begin
  real := ap_data_get_struct(This);
  node := GetNode(vnode);
  ret := real.AddChildFirst(node, STR2CSTR(str));
  result := TreeNode_alloc(cTreeNode, ret);
  rb_iv_set(result, '@owner', This);
  rb_ary_push(rb_iv_get(This, '@items'), result);
end;

function TreeNodes_insert(This, vnode, str: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
  node, ret: TTreeNode;
begin
  real := ap_data_get_struct(This);
  if vnode = Qnil then node := nil else node := GetNode(vnode);
  ret := real.Insert(node, STR2CSTR(str));
  result := TreeNode_alloc(cTreeNode, ret);
  rb_iv_set(result, '@owner', This);
  rb_ary_push(rb_iv_get(This, '@items'), result);
end;

procedure Node_delete(This, vnode: Tvalue);
var
  p: PNode;
  node: TTreeNode;
  n: Integer;
begin
  rb_iv_set(vnode, '@owner', Qnil);
  rb_ary_delete(rb_iv_get(This, '@items'), vnode);
  p := GetP(vnode);
  p^.dead := True;
  node := p^.real;
  for n := 0 to node.Count-1 do
    Node_delete(This, Tvalue(node[n].Data));
end;

function TreeNodes_delete(This, vnode: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
  p: PNode;
  node: TTreeNode;
begin
  Node_delete(This, vnode);
  
  p := GetP(vnode);
  p^.dead := True;
  node := p^.real;
  
  rb_gc;
  real := ap_data_get_struct(This);
  real.Delete(node);
  result := Qnil;
end;

function TreeNodes_clear(This: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
  ary: Tvalue;
  ptr: Pvalue;
  len: Integer;
  vnode: Tvalue;
  p: PNode;
begin
  ary := rb_iv_get(This, '@items');
  ptr := ap_ary_ptr(ary);
  len := ap_ary_len(ary);
  while len > 0 do
  begin
    vnode := ptr^;
    p := GetP(vnode);
    p^.dead := True;
    Inc(ptr);
    Dec(len);
  end;
  rb_ary_clear(ary);
  rb_gc;
  real := ap_data_get_struct(This);
  real.Clear;
  result := Qnil;
end;

function TreeNodes_assign(This, v: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
  source: TPersistent;
  ary, dup: Tvalue;
  ptr: Pvalue;
  len: Integer;
begin
  real := ap_data_get_struct(This);
  ap_data_get_object(v, TPersistent, source);
  real.Assign(source);
  ary := rb_iv_get(v, '@items');
  if RTYPE(ary) = T_ARRAY then
  begin
    ptr := ap_ary_ptr(ary);
    len := ap_ary_len(ary);
    dup := rb_iv_get(This, '@items');
    rb_ary_clear(dup);
    while len > 0 do
    begin
      rb_ary_push(dup, ptr^);
      Inc(ptr);
      Dec(len);
    end;
  end;
  result := v;
end;

function TreeNodes_update(This: Tvalue): Tvalue; cdecl;
var
  real: TTreeNodes;
begin
  real := ap_data_get_struct(This);
  real.BeginUpdate;
  try
    result := rb_yield(Qnil);
  finally
    real.EndUpdate;
  end;
end;

procedure Init_TreeNode;
begin
  cTreeNode := rb_define_class_under(mPhi, 'TreeNode', ap_cObject);
  DefineProp(cTreeNode, TTreeNode);
  rb_define_method(cTreeNode, '[]', @TreeNode_aref, 1);
  DefineAttrGet(cTreeNode, 'data', TreeNode_get_data);
  DefineAttrSet(cTreeNode, 'data', TreeNode_set_data);
  DefineAttrGet(cTreeNode, 'count', TreeNode_get_count);
  DefineAttrSet(cTreeNode, 'text', TreeNode_set_text);
  rb_define_method(cTreeNode, 'sort', @TreeNode_sort, 0);
  rb_define_method(cTreeNode, 'assign', @TreeNode_assign, 1);
  rb_define_method(cTreeNode, 'expand', @TreeNode_expand, 1);
  rb_define_method(cTreeNode, 'collapse', @TreeNode_collapse, 1);
  rb_define_attr(cTreeNode, 'text', 1, 0);
  rb_define_attr(cTreeNode, 'index', 1, 0);
  rb_define_attr(cTreeNode, 'level', 1, 0);
  rb_define_attr(cTreeNode, 'parent', 1, 0);
  rb_define_attr(cTreeNode, 'owner', 1, 0);

  cTreeNodes := rb_define_class_under(mPhi, 'TreeNodes', ap_cObject);
  rb_define_method(cTreeNodes, '[]', @TreeNodes_aref, 1);
  DefineAttrGet(cTreeNodes, 'count', TreeNodes_get_count);
  DefineAttrGet(cTreeNodes, 'first_node', TreeNodes_get_first_node);
  rb_define_method(cTreeNodes, 'add', @TreeNodes_add, 2);
  rb_define_method(cTreeNodes, 'add_first', @TreeNodes_add_first, 2);
  rb_define_method(cTreeNodes, 'add_child', @TreeNodes_add_child, 2);
  rb_define_method(cTreeNodes, 'add_child_first', @TreeNodes_add_child_first, 2);
  rb_define_method(cTreeNodes, 'insert', @TreeNodes_insert, 2);
  rb_define_method(cTreeNodes, 'clear', @TreeNodes_clear, 0);
  rb_define_method(cTreeNodes, 'delete', @TreeNodes_delete, 1);
  rb_define_method(cTreeNodes, 'assign', @TreeNodes_assign, 1);
  rb_define_method(cTreeNodes, 'update', @TreeNodes_update, 0);
  DefineAttrGet(cTreeNodes, 'owner', TreeNodes_get_owner);
end;

exports
  ap_cTreeNode,
  ap_cTreeNodes,
  TreeNode_alloc,
  TreeNodes_alloc;

end.
