unit uStrings;

interface

uses Classes, Rubies;

var
  cStrings, cStringList: Tvalue;

function ap_cStrings: Tvalue; stdcall;
function Strings_alloc(This: Tvalue; real: TStrings): Tvalue; stdcall;
procedure Init_Strings;

implementation

uses uIntern, uAlloc, uProp, uPhi, uConv;

function ap_cStrings: Tvalue; stdcall;
begin
  result := cStrings;
end;

function Strings_alloc(This: Tvalue; real: TStrings): Tvalue; stdcall;
begin
  result := TmpAlloc(This, real);
end;

function StringList_new(argc: integer; argv: Pointer; This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := TStringList.Create;
  result := ObjAlloc(This, real);
  rb_obj_call_init(result, argc, argv);
end;

function Strings_aref(This, v: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  i: Integer;
  S: string;
begin
  real := ap_data_get_struct(This);
  case RTYPE(v) of
  T_FIXNUM:
    begin
      i := FIX2INT(v);
      if i >= real.Count then
        ap_raise(ap_eIndexError, sOut_of_range);
      result := ap_String(real[i]);
    end;
  T_STRING:
    begin
      S := STR2CSTR(v);
      result := ap_String(real.Values[S]);
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
    result := Qnil; // avoid warning
  end;
end;

function Strings_aset(This, v, str: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  i: Integer;
  S: string;
  cstr: string;
begin
  real := ap_data_get_struct(This);
  cstr := STR2CSTR(str);
  case RTYPE(v) of
  T_FIXNUM:
    begin
      i := FIX2INT(v);
      if i >= real.Count then
        ap_raise(ap_eIndexError, sOut_of_range);
      real[i] := cstr;
    end;
  T_STRING:
    begin
      S := STR2CSTR(v);
      real.Values[S] := cstr;
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
  end;
  result := str;
end;

function Strings_get_count(This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.Count);
end;

function Strings_get_text(This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  result := ap_String(real.Text);
end;

function Strings_set_text(This, v: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  real.Text := dl_String(v);
  result := v;
end;

function Strings_get_comma_text(This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  result := ap_String(real.CommaText);
end;

function Strings_set_comma_text(This, v: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  real.CommaText := dl_String(v);
  result := v;
end;

function Strings_get_hash(This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  argc: Integer;
  i: Integer;
  name: string;
begin
  real := ap_data_get_struct(This);
  argc := real.Count;
  result := rb_hash_new;
  for i := 0 to argc-1 do
  begin
    name := real.Names[i];
    rb_hash_aset(result,
      rb_str_new2(PChar(name)),
      rb_str_new2(PChar(real.Values[name]))
    );
  end;
end;

function Strings_equals(argc: integer; argv: Pointer; This: Tvalue): Tvalue; cdecl;
var
  real, other: TStrings;
begin
  other := nil;
  case argc of
  1: other := ap_data_get_struct(Pvalue(argv)^);
  else
    ap_raise(ap_eArgError, sWrong_num_of_args);
  end;
  real := ap_data_get_struct(This);
  result := ap_bool(real.Equals(other));
end;

function Strings_add(argc: integer; argv: Pointer; This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  args: array of Tvalue;
  n: Integer;
  S: string;
begin
  if argc < 1 then
    ap_raise(ap_eArgError, sToo_few_args);
  SetLength(args, argc);
  args := argv;
  S := dl_String(args[0]);
  real := ap_data_get_struct(This);
  if argc > 1 then
  begin
    n := real.AddObject(S, TObject(args[1]));
    rb_gc_mark(real.Objects[n]);
    result := INT2FIX(n)
  end
  else
    result := INT2FIX(real.Add(S));
end;

function Strings_add_strings(This, v: Tvalue): Tvalue; cdecl;
var
  real, other: TStrings;
begin
  real := ap_data_get_struct(This);
  ap_data_get_object(v, TStrings, other);
  real.AddStrings(other);
  result := Qnil;
end;

function Strings_insert(argc: integer; argv: Pointer; This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  args: array of Tvalue;
  n: Integer;
  S: string;
begin
  if argc < 2 then
    ap_raise(ap_eArgError, sToo_few_args);
  SetLength(args, argc);
  args := argv;
  real := ap_data_get_struct(This);
  n := NUM2INT (args[0]);
  S := STR2CSTR(args[1]);
  if (n < 0) or (real.Count <= n) then
    ap_raise(ap_eIndexError, sOut_of_range);
  if argc > 2 then
  begin
    real.InsertObject(n, S, TObject(args[1]));
    rb_gc_mark(real.Objects[n]);
  end
  else
    real.Insert(n, S);
  result := Qnil;
end;

function Strings_delete(This, index: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  real.Delete(NUM2INT(index));
  result := Qnil;
end;

function Strings_clear(argc: integer; argv: Pointer; This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  real.Clear;
  result := Qnil;
end;

function Strings_object_at(This, index: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  result := Tvalue(real.Objects[NUM2INT(index)]);
end;

function Strings_index_of(This, str: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.IndexOf(STR2CSTR(str)));
end;

function Strings_index_of_name(This, str: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.IndexOfName(STR2CSTR(str)));
end;

function Strings_index_of_object(This, obj: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.IndexOfObject(TObject(obj)));
end;

function Strings_load(This, v: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  stream: TStream;
begin
  real := ap_data_get_struct(This);
  case RTYPE(v) of
  T_STRING:
    try
      real.LoadFromFile(STR2CSTR(v));
    except
      on E: EFOpenError do
        ap_raise(ap_eIOError, E.message);
    end;
  T_DATA:
    try
      ap_data_get_object(v, TStream, stream);
      real.LoadFromStream(stream);
    except
      on E: EReadError do;
    end;
  end;
  result := This;
end;

function Strings_save(This, v: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  stream: TStream;
begin
  real := ap_data_get_struct(This);
  case RTYPE(v) of
  T_STRING:
    try
      real.SaveToFile(STR2CSTR(v));
    except
      on E: EFCreateError do
        ap_raise(ap_eIOError, E.message);
    end;
  T_DATA:
    try
      ap_data_get_object(v, TStream, stream);
      real.SaveToStream(stream);
    except
      on E: EWriteError do;
    end;
  end;
  result := v;
end;

function Strings_to_s(This: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  klass: Tvalue;
  i: Integer;
begin
  real := TStrings(ap_data_get_struct(This));

  result := rb_str_new2('#');
  rb_str_cat(result, '<', 1);

  klass := CLASS_OF(This);
  klass := rb_class_path(klass);
  rb_str_concat(result, klass);
  rb_str_cat(result, ': ', 2);

  for i := 0 to real.Count-1 do
  begin
    if i > 0 then
      rb_str_cat(result, ', ', 2);
    rb_str_cat(result, '"', 1);
    ap_str_cat(result, real[i]);
    rb_str_cat(result, '"', 1);
    if real.Objects[i] <> nil then
    begin
      rb_str_cat(result, '=>', 2);
      ap_str_cat(result, ap_str_ptr(rb_inspect(Tvalue(real.Objects[i]))));
    end;
  end;

  rb_str_cat(result, '>', 1);
end;

function Strings_assign(This, v: Tvalue): Tvalue; cdecl;
var
  real: TStrings;
  source: TPersistent;
begin
  real := ap_data_get_struct(This);
  ap_data_get_object(v, TPersistent, source);
  real.Assign(source);
  result := v;
end;

function Strings_find(This, str: Tvalue): Tvalue; cdecl;
var
  real: TStringList;
  i: Integer;
begin
  real := TStringList(ap_data_get_struct(This));
  if real.Find(STR2CSTR(str), i) then
    result := INT2FIX(i)
  else
    result := Qnil
  ;
end;

procedure Init_Strings;
begin
  cStrings := rb_define_class_under(mPhi, 'Strings', ap_cObject);
  // This 'DefineProp' has no effect. because TString is not TClass.
  DefineProp(cStrings, TStrings);

  rb_undef_method(CLASS_OF(cStrings), 'new');

  { properties }
  // vcl: strings, values
  rb_define_method(cStrings, '[]', @Strings_aref, 1);
  // vcl: strings, values
  rb_define_method(cStrings, '[]=', @Strings_aset, 2);

  DefineAttrGet(cStrings, 'count', Strings_get_count);
  DefineAttrGet(cStrings, 'text', Strings_get_text);
  DefineAttrSet(cStrings, 'text', Strings_set_text);
  DefineAttrGet(cStrings, 'comma_text', Strings_get_comma_text);
  DefineAttrSet(cStrings, 'comma_text', Strings_set_comma_text);

  // vcl: names, values
  DefineAttrGet(cStrings, 'hash', Strings_get_hash);
  { methods }
  DefineMethod(cStrings, 'equals', Strings_equals);
  DefineMethod(cStrings, '==', Strings_equals);
  DefineMethod(cStrings, '===', Strings_equals);
  DefineMethod(cStrings, 'add', Strings_add);
  rb_define_method(cStrings, 'add_strings', @Strings_add_strings, 1);
  DefineMethod(cStrings, 'insert', Strings_insert);
  rb_define_method(cStrings, 'delete', @Strings_delete, 1);
  DefineMethod(cStrings, 'clear', Strings_clear);
  rb_define_method(cStrings, 'object_at', @Strings_object_at, 1);
  rb_define_method(cStrings, 'index_of', @Strings_index_of, 1);
  rb_define_method(cStrings, 'index_of_name', @Strings_index_of_name, 1);
  rb_define_method(cStrings, 'index_of_object', @Strings_index_of_object, 1);
  // vcl: load_from_file, load_form_stream
  rb_define_method(cStrings, 'load', @Strings_load, 1);
  // vcl: save_to_file, save_to_stream
  rb_define_method(cStrings, 'save', @Strings_save, 1);
  rb_define_method(cStrings, 'assign', @Strings_assign, 1);
  rb_define_method(cStrings, 'find', @Strings_find, 1);

  DefineAttrGet(cStrings, 'to_s', Strings_to_s);

  cStringList := rb_define_class_under(mPhi, 'StringList', cStrings);
  DefineSingletonMethod(cStringList, 'new', StringList_new);
end;

exports
  ap_cStrings,
  Strings_alloc;

end.
