unit uListItem;

interface

uses Rubies, QComCtrls;

var
  cListItem, cListItems: Tvalue;

function GetItem(obj: Tvalue): TListItem;
function ap_cListItem: Tvalue; stdcall;
function ListItem_alloc(This: Tvalue; real: TListItem): Tvalue; stdcall;
function ap_cListItems: Tvalue; stdcall;
function ListItems_alloc(This: Tvalue; real: TListItems): Tvalue; stdcall;
procedure Init_ListItem;

implementation

uses
  SysUtils, Classes,
  uIntern, uHandle, uAlloc, uProp, uPhi, uConv,
  uSizeConstraints, uStrings,
  uComponent, uControl;

type
  PItem = ^TItem;
  TItem = record
    real: TListItem;
    dead: Boolean;
    data: Tvalue;
  end;

procedure ListItem_free(p: PItem); cdecl;
begin
  dispose(p);
end;

procedure ListItem_mark(p: PItem); cdecl;
begin
  rb_gc_mark(Pointer(p^.data));
end;

function ListItem_alloc1(klass: Tvalue; real: TListItem): Tvalue; stdcall;
var
  p: PItem;
begin
  if real = nil then begin result := Qnil; exit; end;
  new(p);
  p^.real := real;
  p^.dead := False;
  p^.data := Qnil;
  result := rb_data_object_alloc(klass, p, @ListItem_mark, @ListItem_free);
  rb_iv_set(result, '@events', rb_hash_new);
  real.Data := Pointer(result);
end;

function GetP(obj: Tvalue): PItem;
begin
  if rb_obj_is_instance_of(obj, cListItem) = 0 then
    ap_raise(ap_eArgError, sWrong_arg_type);
  result := ap_data_get_struct(obj);
end;

function GetItem(obj: Tvalue): TListItem;
var
  p: PItem;
begin
  p := GetP(obj);
  if p^.dead then
  begin
    ap_raise(ap_eArgError, 'dead item');
    result := nil;
  end
  else
    result := p^.real;
end;

function ap_cListItem: Tvalue; stdcall;
begin
  result := cListItem;
end;

function ap_cListItems: Tvalue; stdcall;
begin
  result := cListItems;
end;

procedure ListItem_setup(obj: Tvalue; real: TListItem);
begin
  rb_iv_set(obj, '@sub_items', Strings_alloc(ap_cStrings, real.SubItems));
end;

function ListItem_alloc(This: Tvalue; real: TListItem): Tvalue; stdcall;
begin
  result := ListItem_alloc1(This, real);
  ListItem_setup(result, real);
end;

function ListItem_get_data(This: Tvalue): Tvalue; cdecl;
var
  p: PItem;
begin
  p := GetP(This);
  result := p^.data;
end;

function ListItem_set_data(This, v: Tvalue): Tvalue; cdecl;
var
  p: PItem;
begin
  p := GetP(This);
  p^.data := v;
  result := v;
end;

function ListItem_get_caption(This: Tvalue): Tvalue; cdecl;
var
  real: TListItem;
begin
  real := GetItem(This);
  result := ap_String(real.Caption);
end;

function ListItem_set_caption(This, v: Tvalue): Tvalue; cdecl;
var
  real: TListItem;
begin
  real := GetItem(This);
  real.Caption := STR2CSTR(v);
  result := v;
end;

function ListItem_get_list_view(This: Tvalue): Tvalue; cdecl;
var
  real: TListItem;
begin
  real := GetItem(This);
  if real.ListView = nil then
    result := Qnil
  else
    result := real.ListView.tag
  ;
end;

function ListItem_get_index(This: Tvalue): Tvalue; cdecl;
var
  real: TListItem;
begin
  real := GetItem(This);
  result := INT2FIX(real.Index);
end;

procedure ListItems_setup(obj: Tvalue; real: TListItems); stdcall;
begin
  rb_iv_set(obj, '@items', rb_ary_new);
end;

function ListItems_alloc(This: Tvalue; real: TListItems): Tvalue; stdcall;
begin
  result := TmpAlloc(This, real);
  ListItems_setup(result, real);
end;

function ListItems_aref(This, index: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
  n: Integer;
begin
  real := ap_data_get_struct(This);
  n := FIX2INT(index);
  if (n < 0) or (real.Count <= n) then
    ap_raise(ap_eIndexError, sOut_of_range);
  if real[n].Data = nil then
    result := Qnil
  else
    result := Tvalue(real[n].Data);
end;

function ListItems_get_count(This: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.Count);
end;

function ListItems_set_count(This, v: Tvalue): Tvalue; cdecl;
begin
  result := v;
end;

function ListItems_get_owner(This: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
begin
  real := ap_data_get_struct(This);
  if real.Owner = nil then
    result := Qnil
  else
    result := real.Owner.tag
  ;
end;

function ListItems_add(This: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
begin
  real := ap_data_get_struct(This);
  result := ListItem_alloc(cListItem, real.Add);
  rb_iv_set(result, '@owner', This);
  rb_ary_push(rb_iv_get(This, '@items'), result);
end;

function ListItems_insert(This, index: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
  n: Integer;
begin
  real := ap_data_get_struct(This);
  n := FIX2INT(index);
  if (n < 0) or (real.Count < n) then
    ap_raise(ap_eIndexError, sOut_of_range);
  result := ListItem_alloc(cListItem, real.Insert(n));
  rb_iv_set(result, '@owner', This);
  rb_ary_push(rb_iv_get(This, '@items'), result);
end;

function ListItems_delete(This, index: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
  n: Integer;
  vitem: Tvalue;
  p: PItem;
begin
  real := ap_data_get_struct(This);
  n := FIX2INT(index);
  if (n < 0) or (real.Count <= n) then
    ap_raise(ap_eIndexError, sOut_of_range);
  vitem := Tvalue(real[n].Data);
  p := GetP(vitem);
  p^.dead := True;
  rb_ary_delete(rb_iv_get(This, '@items'), vitem);
  rb_gc;
  real.Delete(n);
  result := Qnil;
end;

function ListItems_clear(This: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
  ary: Tvalue;
  ptr: Pvalue;
  len: Integer;
  vitem: Tvalue;
  p: PItem;
begin
  ary := rb_iv_get(This, '@items');
  ptr := ap_ary_ptr(ary);
  len := ap_ary_len(ary);
  while len > 0 do
  begin
    vitem := ptr^;
    p := GetP(vitem);
    p^.dead := True;
    Inc(ptr);
    Dec(len);
  end;
  rb_ary_clear(ary);
  rb_gc;
  real := ap_data_get_struct(This);
  real.Clear;
  result := Qnil;
end;

function ListItems_index_of(This, v: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
  item: TListItem;
begin
  real := ap_data_get_struct(This);
  ap_data_get_object(v, TListItem, item);
  result := INT2FIX(real.IndexOf(item));
end;

function ListItems_update(This: Tvalue): Tvalue; cdecl;
var
  real: TListItems;
begin
  real := ap_data_get_struct(This);
  real.BeginUpdate;
  try
    result := rb_yield(Qnil);
  finally
    real.EndUpdate;
  end;
end;

procedure Init_ListItem;
begin
  cListItem := rb_define_class_under(mPhi, 'ListItem', ap_cObject);
  DefineProp(cListItem, TListItem);
  DefineAttrGet(cListItem, 'data', ListItem_get_data);
  DefineAttrSet(cListItem, 'data', ListItem_set_data);
  DefineAttrGet(cListItem, 'caption', ListItem_get_caption);
  DefineAttrSet(cListItem, 'caption', ListItem_set_caption);
  DefineAttrGet(cListItem, 'list_view', ListItem_get_list_view);
  DefineAttrGet(cListItem, 'index', ListItem_get_index);
  rb_define_attr(cListItem, 'sub_items', 1, 0);
  rb_define_attr(cListItem, 'owner', 1, 0);

  cListItems := rb_define_class_under(mPhi, 'ListItems', ap_cObject);
  DefineProp(cListItems, TListItems);
  rb_define_method(cListItems, '[]', @ListItems_aref, 1);
  DefineAttrGet(cListItems, 'count', ListItems_get_count);
  DefineAttrSet(cListItems, 'count', ListItems_set_count);
  DefineAttrGet(cListItems, 'owner', ListItems_get_owner);
  rb_define_method(cListItems, 'add', @ListItems_add, 0);
  rb_define_method(cListItems, 'insert', @ListItems_insert, 1);
  rb_define_method(cListItems, 'delete', @ListItems_delete, 1);
  rb_define_method(cListItems, 'clear', @ListItems_clear, 0);
  rb_define_method(cListItems, 'index_of', @ListItems_index_of, 1);
  rb_define_method(cListItems, 'update', @ListItems_update, 0);
end;

exports
  ap_cListItem,
  ap_cListItems,
  ListItem_alloc,
  ListItems_alloc;

end.
