unit uDateTime;

interface

uses Rubies;

var
  LOCAL_TIME_DIFF : double = 0.0;//DAY
  JD_DIFF         : double = 2415019.0;//DAY
  TIME_T_DIFF     : double = 2209194000.0;//SEC

type
  PDateTime = ^TDateTime;

var
  cDateTime: Tvalue;

function ap_DateTime(v: TDateTime): Tvalue; stdcall;
function dl_DateTime(v: Tvalue): TDateTime; stdcall;

function ap_cDateTime: Tvalue; stdcall;
function DateTime_alloc(This: Tvalue; real: TDateTime): Tvalue; stdcall;
procedure Init_DateTime;

implementation

uses Types, SysUtils, uAlloc, uIntern, uPhi, uConv;

function ap_DateTime(v: TDateTime): Tvalue;
begin
  result := DateTime_alloc(cDateTime, v);
end;

function dl_DateTime(v: Tvalue): TDateTime;
var
  class_name: string;
begin
  result := -1;
  case RTYPE(v) of
  T_STRING:
    try
      result := StrToDateTime(STR2CSTR(v));
    except
      on E: EConvertError do
        ap_raise(ap_eArgError, E.message);
    end;
  T_OBJECT:
    begin
      class_name := rb_class2name(CLASS_OF(v));
      if class_name = 'Date' then
        result := 
          dl_Double(rb_funcall2(v, rb_intern('jd'), 0, nil)) - jd_diff
      else
        ap_raise(ap_eArgError, sWrong_arg_type);
    end;
  T_DATA:
    begin
      class_name := rb_class2name(CLASS_OF(v));
      if class_name = 'Time' then
        result := 
        (
          dl_Integer(rb_funcall2(v, rb_intern('to_i'), 0, nil)) + Time_T_Diff
        ) / SecsPerDay
      else if ap_kind_of(v, cDateTime) then
        result := PDateTime(ap_data_get_struct(v))^
      else
        ap_raise(ap_eArgError, sWrong_arg_type);
    end;
  else
      ap_raise(ap_eArgError, sWrong_arg_type);
  end;
end;

function ap_cDateTime: Tvalue; stdcall;
begin
  result := cDateTime;
end;

function DateTime_alloc(This: Tvalue; real: TDateTime): Tvalue; stdcall;
var
  p: PDateTime;
begin
  new(p);
  result := rb_data_object_alloc(This, p, nil, @ap_dispose);
  p^ := real;
end;

function DateTime_new(argc: Integer; argv: Pointer; This: Tvalue): Tvalue; cdecl;
var
  args: array of Tvalue;
  p: PDateTime;
  y, m, d, h, n, s, z: word;
begin
  SetLength(args, argc);
  args := argv;
  new(p);
  result := rb_data_object_alloc(This, p, nil, @ap_dispose);
  case argc of
  0:
    p^ := Now;
  1:
    p^ := dl_DateTime(args[0]);
  3..7:
    try
      y := FIX2INT(args[0]);
      m := FIX2INT(args[1]);
      d := FIX2INT(args[2]);
      h := 0 ; if argc > 3 then h := FIX2INT(args[3]);
      n := 0 ; if argc > 4 then n := FIX2INT(args[4]);
      s := 0 ; if argc > 5 then s := FIX2INT(args[5]);
      z := 0 ; if argc > 6 then z := FIX2INT(args[6]);
      if (y=0) and (m=0) and (d=0)
      then
        p^ := encodeTime(h, n, s, z)
      else
        p^ := encodeDate(y, m, d) + encodeTime(h, n, s, z);
    except
      on E: Exception do
        ap_raise(ap_eArgError, E.message);
    end;
  else
    ap_raise(ap_eArgError, sWrong_num_of_args);
  end;
  rb_obj_call_init(result, argc, argv);
end;

function DateTime_to_s(This: Tvalue): Tvalue; cdecl;
var
  real: TDateTime;
begin
  real := PDateTime(ap_data_get_struct(This))^;
  if real < 1 then
    result := rb_str_new2(PChar(TimeToStr(real)))
  else
    result := rb_str_new2(PChar(DateTimeToStr(real)));
end;

function DateTime_format(This, f: Tvalue): Tvalue; cdecl;
var
  real: TDateTime;
begin
  result := Qnil;
  try
    real := PDateTime(ap_data_get_struct(This))^;
    result := rb_str_new2(PChar(FormatDateTime(STR2CSTR(f), real)));
  except
    on E: Exception do
      ap_raise(ap_eArgError, E.message);
  end;
end;

procedure apDecodeDateTime(v: Tvalue; var y, m, d, h, n, s, z: word);
var
  real: TDateTime;
begin
  try
    real := PDateTime(ap_data_get_struct(v))^;
    DecodeDate(real, y, m, d);
    DecodeTime(real, h, n, s, z);
  except
    on E: Exception do
      ap_raise(ap_eArgError, E.message);
  end;
end;

procedure apEncodeDateTime(v: Tvalue; y, m, d, h, n, s, z: word);
begin
  try
    PDateTime(ap_data_get_struct(v))^ :=
      EncodeDate(y, m, d) + EncodeTime(h, n, s, 0);
  except
    on E: Exception do
      ap_raise(ap_eArgError, E.message);
  end;
end;

function DateTime_set_year(This, v: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  result := v;
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  apEncodeDateTime(This, FIX2INT(v), m, d, h, n, s, z);
end;

function DateTime_get_year(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := INT2FIX(y);
end;

function DateTime_set_month(This, v: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  result := v;
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  apEncodeDateTime(This, y, FIX2INT(v), d, h, n, s, z);
end;

function DateTime_get_month(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := INT2FIX(m);
end;

function DateTime_set_day(This, v: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  result := v;
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  apEncodeDateTime(This, y, m, FIX2INT(v), h, n, s, z);
end;

function DateTime_get_day(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := INT2FIX(d);
end;

function DateTime_set_hour(This, v: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  result := v;
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  apEncodeDateTime(This, y, m, d, FIX2INT(v), n, s, z);
end;

function DateTime_get_hour(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := INT2FIX(h);
end;

function DateTime_set_minute(This, v: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  result := v;
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  apEncodeDateTime(This, y, m, d, h, FIX2INT(v), s, z);
end;

function DateTime_get_minute(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := INT2FIX(n);
end;

function DateTime_set_second(This, v: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  result := v;
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  apEncodeDateTime(This, y, m, d, h, n, FIX2INT(v), z);
end;

function DateTime_get_second(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := INT2FIX(s);
end;

function DateTime_set_msec(This, v: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  result := v;
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  apEncodeDateTime(This, y, m, d, h, n, s, FIX2INT(v));
end;

function DateTime_get_msec(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := INT2FIX(z);
end;

function DateTime_to_a(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := rb_ary_new;
  rb_ary_push(result, INT2FIX(y));
  rb_ary_push(result, INT2FIX(m));
  rb_ary_push(result, INT2FIX(d));
  rb_ary_push(result, INT2FIX(h));
  rb_ary_push(result, INT2FIX(n));
  rb_ary_push(result, INT2FIX(s));
  rb_ary_push(result, INT2FIX(z));
end;

function DateTime_add(This, v: Tvalue): Tvalue; cdecl;
begin
  result := ap_DateTime(dl_DateTime(This) + dl_Double(v));
end;

function DateTime_sub(This, v: Tvalue): Tvalue; cdecl;
var
  real: TDateTime;
begin
  real := dl_DateTime(This);
  if ap_kind_of(v, cDateTime) then
    result := ap_Float   (real - dl_DateTime(v))
  else
    result := ap_DateTime(real - dl_Double  (v));
end;

function DateTime_cmp(This, v: Tvalue): Tvalue; cdecl;
var
  l, r: TDateTime;
begin
  l := dl_DateTime(This);
  r := dl_DateTime(v);
  if l > r then result := ap_Fixnum(+1) else
  if l < r then result := ap_Fixnum(-1) else
                result := ap_Fixnum( 0);
end;

function DateTime_eqq(This, v: Tvalue): Tvalue; cdecl;
var
  l, r: TDateTime;
begin
  l := dl_DateTime(This);
  r := dl_DateTime(v);
  if l = r then result := Qtrue else result := Qfalse;
end;

function DateTime_inspect(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_String(
    '#<'+dl_ClassName(This)+': '+
    DateTimeToStr(dl_DateTime(This))+'>');
end;

function DateTime_today(This: Tvalue): Tvalue; cdecl;
begin
  result := DateTime_alloc(This, date );
end;

function DateTime_get_time_t(This: Tvalue): Tvalue; cdecl;
begin
  result := rb_dbl2big(dl_DateTime(This)*SecsPerDay - Time_T_Diff + 0.0001);
end;

function DateTime_get_jd(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_Float(dl_DateTime(This)+jd_diff);
end;

function Phi_get_date_separator(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_String( DateSeparator );
end;

function Phi_set_date_separator(This, v: Tvalue): Tvalue; cdecl;
var
  s:string;
begin
  result := Qnil;
  s := dl_String(v);
  if length(s)=1 then begin
    DateSeparator := s[1];
    result := v;
  end
  else begin
      ap_raise(ap_eArgError, sWrong_arg_type);
  end;
end;

function Phi_get_time_separator(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_String( TimeSeparator );
end;

function Phi_set_time_separator(This, v: Tvalue): Tvalue; cdecl;
var
  s:string;
begin
  result := Qnil;
  s := dl_String(v);
  if length(s)=1 then begin
    TimeSeparator := s[1];
    result := v;
  end
  else begin
      ap_raise(ap_eArgError, sWrong_arg_type);
  end;
end;

(*
=begin

Phi#date_format
Phi#time_format

DateTime#to_s, DBGrid ł̓t\̑ DateTime 𕶎
ϊKvꍇɎgptH[}bgB

Windows[Rg[pl]-[n]-[t]-[Z`]
[]-[Ԃ̌`]Ŏw肵̂ŏɓĂB

t\̓AvP[V̉ʐ݌vɑ傢ɉêŁC
AvP[VŎw肷ׂƎvB

Delphi  ShortDateFormat  LongTimeFormat Ƃbv́B
(Short  Long ƂɂȂĂ邱Ƃɒ)B

Phi#short_date_format  Phi#long_time_format  alias łB

--

Phi#short_date_format
Phi#short_time_format
Phi#long_time_format
Phi#long_date_format

DateTime#format ł
    "c"         (ShortDateFormat { LongTimeFormat)
    "ddddd"     (ShortDateFormat)
    "tt"        (LongTimeFormat)
    "dddddd"    (LongDateFormat)
    "t"         (ShortTimeFormat)

----

Phi#date_format
Phi#time_format

Default format used by DateTime#to_s,  DBGrid  or  other  convertion
from DateTime to String.

User can design them by setting in "Locale" of  "Control  Panel"  of
Windows.

However, I  would  recommend  that  Application  must  preset  them, 
because format of date &  time  has  much  effect  to  Application's 
Screen Design.

They are the aliases of
    Phi#short_date_format and Phi#long_time_format,
  and the wraps of
    Delphi's ShortDateFormat and LongTimeFormat.

--

Phi#short_date_format
Phi#short_time_format
Phi#long_time_format
Phi#long_date_format

In DateTime#format
    "c"         (ShortDateFormat + LongTimeFormat)
    "ddddd"     (ShortDateFormat)
    "tt"        (LongTimeFormat)
    "dddddd"    (LongDateFormat)
    "t"         (ShortTimeFormat)

=end
*)

function Phi_get_short_date_format(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_String( ShortDateFormat );
end;

function Phi_set_short_date_format(This, v: Tvalue): Tvalue; cdecl;
begin
  result := v;
  ShortDateFormat := dl_String(v);
end;

function Phi_get_long_date_format(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_String( LongDateFormat );
end;

function Phi_set_long_date_format(This, v: Tvalue): Tvalue; cdecl;
begin
  result := v;
  LongDateFormat := dl_String(v);
end;

function Phi_get_long_time_format(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_String( LongTimeFormat );
end;

function Phi_set_long_time_format(This, v: Tvalue): Tvalue; cdecl;
begin
  result := v;
  LongTimeFormat := dl_String(v);
end;

function Phi_get_short_time_format(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_String( ShortTimeFormat );
end;

function Phi_set_short_time_format(This, v: Tvalue): Tvalue; cdecl;
begin
  result := v;
  ShortTimeFormat := dl_String(v);
end;

(*
=begin

Delphi  Date, Time ͌ݎ̓tEԂO[o֐Ȃ̂

  date = DateTime.date
  time = DateTime.time

ƂȂ͂ł邪B

Apollo ł̓CX^X̃\bhƂBCӂ DateTime CX^X
tE𒊏ołBL̋Lq͎̂悤ɂȂB

  date = DateTime.new.date
  time = DateTime.new.time

=end
*)

function DateTime_date(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
  real: TDateTime;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  real := EncodeDate(y, m, d);
  result := DateTime_alloc(cDateTime, real);
end;

function DateTime_time(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
  real: TDateTime;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  real := EncodeTime(h, n, s, 0);
  result := DateTime_alloc(cDateTime, real);
end;

function DateTime_inc_month(argc: Integer; argv: Pointer; This: Tvalue): Tvalue; cdecl;
var
  args: array of Tvalue;
  i: Integer;
begin
  SetLength(args, argc);
  args := argv;
  if argc = 1 then
    i := NUM2INT(args[0])
  else
    i := 1;
  result := ap_DateTime(IncMonth(dl_DateTime(This), i));
end;

function DateTime_day_of_week(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_Fixnum(DayOfWeek(dl_DateTime(This)));
end;

function DateTime_end_of_month(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := ap_DateTime(EncodeDate(y, m, MonthDays[IsLeapYear(y), m]) + EncodeTime(23, 59, 59, 999));
end;

function DateTime_end_of_year(This: Tvalue): Tvalue; cdecl;
var
  y, m, d, h, n, s, z: word;
begin
  apDecodeDateTime(This, y, m, d, h, n, s, z);
  result := ap_DateTime(EncodeDate(y, 12, 31) + EncodeTime(23, 59, 59, 999));
end;
(*
function LocalTimeDiff: Double;
var
  sys1, loc, sys2: TSystemTime;
  sysDateTime, locDateTime: TDateTime;
  i: Integer;
begin
  for i := 1 to 10 do
  begin
    GetSystemTime(sys1);
    GetLocalTime(loc);
    GetSystemTime(sys2);
    if sys2.wSecond = sys1.wSecond then break;
  end;
  with sys1 do SysDateTime := encodeDate(wYear, wMonth, wDay)
                             +encodeTime(wHour, wMinute, wSecond, 0);
  with loc  do LocDateTime := encodeDate(wYear, wMonth, wDay)
                             +encodeTime(wHour, wMinute, wSecond, 0);
  result := LocDateTime - SysDateTime;
end;
*)
procedure Init_DateTime;
begin
  LOCAL_TIME_DIFF := 0;//DAY
  TIME_T_DIFF     := 2209161600.0 + LOCAL_TIME_DIFF*SecsPerDay;
  //
  //  not correct JD_DIFF, because Date do not support UTC.
  //
  //JD_DIFF         : double = 2415019.0 + LOCAL_TIME_DIFF;

  ap_define_const(mPhi, 'SecsPerDay', INT2FIX(SecsPerDay));
  ap_define_const(mPhi, 'MSecsPerDay', INT2FIX(MSecsPerDay));
  ap_define_const(mPhi, 'LocalTimeDiff', ap_Float(LOCAL_TIME_DIFF));

  cDateTime := rb_define_class_under(mPhi, 'DateTime', ap_cObject);
  DefineSingletonMethod(cDateTime, 'new', DateTime_new);
  DefineSingletonMethod(cDateTime, 'now', DateTime_new);
  DefineAttrSet(cDateTime, 'year', DateTime_set_year);
  DefineAttrGet(cDateTime, 'year', DateTime_get_year);
  DefineAttrSet(cDateTime, 'month', DateTime_set_month);
  DefineAttrGet(cDateTime, 'month', DateTime_get_month);
  DefineAttrSet(cDateTime, 'day', DateTime_set_day);
  DefineAttrGet(cDateTime, 'day', DateTime_get_day);
  DefineAttrSet(cDateTime, 'hour', DateTime_set_hour);
  DefineAttrGet(cDateTime, 'hour', DateTime_get_hour);
  DefineAttrSet(cDateTime, 'minute', DateTime_set_minute);
  DefineAttrGet(cDateTime, 'minute', DateTime_get_minute);
  DefineAttrSet(cDateTime, 'second', DateTime_set_second);
  DefineAttrGet(cDateTime, 'second', DateTime_get_second);

  DefineAttrSet(cDateTime, 'msec', DateTime_set_msec);
  DefineAttrGet(cDateTime, 'msec', DateTime_get_msec);
  DefineAttrGet(cDateTime, 'to_s', DateTime_to_s);
  DefineAttrGet(cDateTime, 'to_a', DateTime_to_a);
  rb_define_method(cDateTime, 'format', @DateTime_format, 1);

  rb_define_method(cDateTime, '+', @DateTime_add, 1);
  rb_define_method(cDateTime, '-', @DateTime_sub, 1);
  rb_define_method(cDateTime, '<=>', @DateTime_cmp, 1);
  rb_include_module(cDateTime, ap_mComparable);
  rb_define_method(cDateTime, '===', @DateTime_eqq, 1);

  rb_define_method(cDateTime, 'inspect', @DateTime_inspect, 0);
  rb_define_singleton_method(cDateTime, 'today', @DateTime_today,0);
  DefineModuleAttrGet(mPhi, 'date_separator', Phi_get_date_separator);
  DefineModuleAttrSet(mPhi, 'date_separator', Phi_set_date_separator);
  DefineModuleAttrGet(mPhi, 'time_separator', Phi_get_time_separator);
  DefineModuleAttrSet(mPhi, 'time_separator', Phi_set_time_separator);
  DefineAttrGet(cDateTime, 'time_t', DateTime_get_time_t);
  DefineAttrGet(cDateTime, 'jd', DateTime_get_jd);

  rb_define_method(cDateTime, 'date', @DateTime_date, 0);
  rb_define_method(cDateTime, 'time', @DateTime_time, 0);
  rb_define_method(cDateTime, 'inc_month', @DateTime_inc_month, -1);
  rb_define_method(cDateTime, 'day_of_week', @DateTime_day_of_week, 0);
  rb_define_method(cDateTime, 'end_of_month', @DateTime_end_of_month, 0);
  rb_define_method(cDateTime, 'end_of_year', @DateTime_end_of_year, 0);

  DefineModuleAttrGet(mPhi, 'short_date_format', Phi_get_short_date_format);
  DefineModuleAttrSet(mPhi, 'short_date_format', Phi_set_short_date_format);
  DefineModuleAttrGet(mPhi, 'short_time_format', Phi_get_short_time_format);
  DefineModuleAttrSet(mPhi, 'short_time_format', Phi_set_short_time_format);
  DefineModuleAttrGet(mPhi, 'long_date_format', Phi_get_long_date_format);
  DefineModuleAttrSet(mPhi, 'long_date_format', Phi_set_long_date_format);
  DefineModuleAttrGet(mPhi, 'long_time_format', Phi_get_long_time_format);
  DefineModuleAttrSet(mPhi, 'long_time_format', Phi_set_long_time_format);

  DefineModuleAttrGet(mPhi, 'date_format', Phi_get_short_date_format);
  DefineModuleAttrSet(mPhi, 'date_format', Phi_set_short_date_format);
  DefineModuleAttrGet(mPhi, 'time_format', Phi_get_long_time_format);
  DefineModuleAttrSet(mPhi, 'time_format', Phi_set_long_time_format);
end;

exports
  ap_DateTime,
  dl_DateTime,
  ap_cDateTime,
  DateTime_alloc;

end.
