unit uParam;

interface

uses Rubies, DB;

var
  cParam, cParams: Tvalue;

function Param_alloc(This: Tvalue; real: TParam): Tvalue;
procedure Init_Param;

implementation

uses
  SysUtils, Classes,
{$IFDEF PHIEMBED}
  PhiMainUnit, uHandle, uAlloc, uProp, uConv, uDateTime,
{$ELSE}
  PhiExternal,
{$ENDIF}
  uRDB;

function Param_alloc(This: Tvalue; real: TParam): Tvalue;
begin
  result := TmpAlloc(This, real);
end;

function Param_assign(This, v: Tvalue): Tvalue; cdecl;
var
  real: TParam;
  source: TPersistent;
begin
  real := ap_data_get_struct(This);
  ap_data_get_object(v, TPersistent, source);
  real.Assign(source);
  result := Qnil;
end;

function Param_load(This, v, blob_t: Tvalue): Tvalue; cdecl;
var
  real: TParam;
  stream: TStream;
  BlobType: TBlobType;
begin
  real := ap_data_get_struct(This);
  BlobType := TBlobType(FIX2INT(blob_t));
  case RTYPE(v) of
  T_STRING:
    try
      real.LoadFromFile(STR2CSTR(v), BlobType);
    except
      on E: EFOpenError do
        ap_raise(ap_eIOError, E.message);
    end;
  T_DATA:
    try
      ap_data_get_object(v, TStream, stream);
      real.LoadFromStream(stream, BlobType);
    except
      on E: EReadError do;
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
  end;
  result := This;
end;

function Param_get_type(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(Ord(real.DataType));
end;

function Param_get_to_f(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  result := rb_float_new(real.AsFloat);
end;

function Param_get_to_i(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.AsInteger);
end;

function Param_get_to_s(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  case real.DataType of
  ftTime:
    result := rb_str_new2(PChar(TimeToStr(real.AsTime)));
  ftDate, ftDateTime:
    result := rb_str_new2(PChar(DateTimeToStr(real.AsDateTime)));
  else
    result := rb_str_new2(PChar(real.AsString));
  end;
end;

function Param_get_name(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  result := rb_str_new2(PChar(real.Name));
end;

function Param_set_as_memo(This,v: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  real.AsMemo := dl_String(v);
  result := v;
end;

function Params_aref(This, p: Tvalue): Tvalue; cdecl;
var
  s: PChar;
  n: Integer;
  real: TParams;
  param: TParam;
begin
  real := ap_data_get_struct(This);
  param := nil;

  case RTYPE(p) of
  T_STRING:
    begin
      s := STR2CSTR(p);
      param := real.ParamByName(s);
    end;
  T_FIXNUM:
    begin
      n := FIX2INT(p);
      if (n < 0) or (real.Count <= n) then
        ap_raise(ap_eIndexError, sOut_of_range);
      param := real[n];
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
  end;

  if param = nil then
    result := Qnil
  else
    result := Param_alloc(cParam, param)
  ;
end;

function Params_aset(This, p, v: Tvalue): Tvalue; cdecl;
var
  n: Integer;
  real: TParams;
  param: TParam;
  klass: Tvalue;
  klass_name: string;
  DateTime: TDateTime;
  kind_p: Tvalue;
begin
  real := ap_data_get_struct(This);
  param := nil;

  case RTYPE(p) of
  T_STRING:
    try
      param := real.ParamByName(STR2CSTR(p));
    except
      on E: Exception do
        ap_raise(ap_eDatabaseError, E.message);
    end;
  T_FIXNUM:
    begin
      n := FIX2INT(p);
      if (n < 0) or (real.Count <= n) then
        ap_raise(ap_eIndexError, sOut_of_range);
      param := real[n];
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
  end;

  if param = nil then
    result := Qnil
  else
  begin
    try
      case RTYPE(v) of
      T_NIL   : param.Clear;
      T_STRING:
        if param.DataType = ftMemo then
          param.AsMemo     := dl_String(v)
        else
          param.AsString   := dl_String(v)
        ;
      T_FIXNUM: param.AsInteger  := FIX2INT(v);
      T_BIGNUM: param.AsCurrency := NUM2INT(v);
      T_FLOAT : param.AsFloat    := NUM2DBL(v);
      T_TRUE  : param.AsBoolean  := True;
      T_FALSE : param.AsBoolean  := False;
      T_DATA:
        begin
          klass := CLASS_OF(v);
          klass := rb_class_path(klass);
          klass_name := STR2CSTR(klass);
          kind_p := rb_obj_is_kind_of(v, ap_cDateTime);
          if RTEST(kind_p) then
          begin
            DateTime := PDateTime(ap_data_get_struct(v))^;
            if DateTime < 1 then
              param.AsTime := DateTime
            else
              param.AsDateTime := DateTime;
          end
          else
            param.AsString := 'Unknown '+klass_name+':class';
        end;
      else
        ap_raise(ap_eArgError, sWrong_arg_type);
      end;
    except
      on E: Exception do
        ap_raise(ap_eDatabaseError, E.message);
    end;
    result := Param_alloc(cParam, param);
  end;
end;

function Params_get_count(This: Tvalue): Tvalue; cdecl;
var
  real: TParams;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.Count);
end;

procedure Init_Param;
begin
  DefineConstSetType(mRDB, TypeInfo(TParamType));
//  DefineConstSetType(mRDB, TypeInfo(TBlobType));

  cParam := rb_define_class_under(mRDB, 'Param', ap_cObject);
  DefineProp(cParam, TParam);
  rb_define_method(cParam, 'assign', @Param_assign, 1);
  rb_define_method(cParam, 'load', @Param_load, 1); // vcl: load_from_file, load_from_stream
  DefineAttrGet(cParam, 'type', Param_get_type); // vcl: DataType
  DefineAttrGet(cParam, 'to_f', Param_get_to_f); // vcl: AsFloat
  DefineAttrGet(cParam, 'to_i', Param_get_to_i); // vcl: AsInteger
  DefineAttrGet(cParam, 'to_s', Param_get_to_s); // vcl: AsString
  DefineAttrGet(cParam, 'name', Param_get_name);
  DefineAttrSet(cParam, 'as_memo', Param_set_as_memo);

  cParams := rb_define_class_under(mRDB, 'Params', ap_cObject);
  rb_define_method(cParams, '[]', @Params_aref, 1);
  rb_define_method(cParams, '[]=', @Params_aset, 2);
  DefineAttrGet(cParams, 'count', Params_get_count);
end;

end.
